/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!              Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalCam                                                            //
//                                                                         //
// Hold the Pedestal information for all pixels in the camera              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MGeomCam.h"

ClassImp(MPedestalCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Creates a TClonesArray of MPedestalPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel. Later, a call to MPedestalCam::InitSize() 
// has to be performed (in MGeomApply). 
//
// Creates a TClonesArray of MPedestalPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel AREA. Later, a call to MPedestalCam::InitAreas() 
// has to be performed (in MGeomApply). 
//
// Creates a TClonesArray of MPedestalPix containers, initialized to 1 entry, destinated
// to hold one container per camera SECTOR. Later, a call to MPedestalCam::InitSectors() 
// has to be performed (in MGeomApply). 
//
MPedestalCam::MPedestalCam(const char *name, const char *title) 
    : fTotalEntries(0)
{
  fName  = name  ? name  : "MPedestalCam";
  fTitle = title ? title : "Storage container for all Pedestal Information in the camera";
  
  fArray            = new TClonesArray("MPedestalPix", 1);
  fAverageAreas     = new TClonesArray("MPedestalPix", 1);
  fAverageSectors   = new TClonesArray("MPedestalPix", 1);
}

// --------------------------------------------------------------------------
//
// Deletes the following TClonesArray's of MPedestalPix containers (if exist):
// - fArray
// - fAverageAreas
// - fAverageSectors
//  
MPedestalCam::~MPedestalCam()
{
  delete fArray;
  delete fAverageAreas;
  delete fAverageSectors;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MPedestalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for:
// - fAverageAreas
//
void MPedestalCam::InitAverageAreas(const UInt_t i)
{
  fAverageAreas->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for:
// - fAverageSectors
//
void MPedestalCam::InitAverageSectors(const UInt_t i)
{
  fAverageSectors->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls:
// - InitSize()
// - InitAverageAreas()
// - InitAverageSectors()
//
void MPedestalCam::Init(const MGeomCam &geom)
{
  InitSize          (geom.GetNumPixels() );
  InitAverageAreas  (geom.GetNumAreas()  );
  InitAverageSectors(geom.GetNumSectors());
}

// --------------------------------------------------------------------------
//
// This function returns the current size of the TClonesArray 
// independently if the MPedestalPix is filled with values or not.
//
// Get the size of the MPedestalCam
//
Int_t MPedestalCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fAverageAreas
// independently if the MPedestalPix is filled with values or not.
//
const Int_t MPedestalCam::GetAverageAreas() const
{
  return fAverageAreas->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fAverageSectors
// independently if the MPedestalPix is filled with values or not.
//
const Int_t MPedestalCam::GetAverageSectors() const
{
  return fAverageSectors->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i)
{
    return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MPedestalPix &MPedestalCam::operator[](Int_t i) const
{
    return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
MPedestalPix &MPedestalCam::GetAverageArea(UInt_t i)
{
  return *static_cast<MPedestalPix*>(fAverageAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
const MPedestalPix &MPedestalCam::GetAverageArea(UInt_t i) const 
{
  return *static_cast<MPedestalPix*>(fAverageAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
MPedestalPix &MPedestalCam::GetAverageSector(UInt_t i)
{
  return *static_cast<MPedestalPix*>(fAverageSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
const MPedestalPix &MPedestalCam::GetAverageSector(UInt_t i) const 
{
  return *static_cast<MPedestalPix*>(fAverageSectors->UncheckedAt(i));
}

// --------------------------------------
//
// Calls the ForEach macro for the TClonesArray fArray with the argument Clear()
// 
// Loops over the fAverageAreas, calling the function Clear() for 
// every entry in fAverageAreas
//
// Loops over the fAverageSectors, calling the function Clear() for 
// every entry in fAverageSectors
// 
void MPedestalCam::Clear(Option_t *o)
{
  fArray->ForEach(TObject, Clear)();
  
  //
  // another ForEach does not compile, thus have to do the loop ourselves:
  //
  for (Int_t i=0;i<GetAverageAreas();i++)
    fAverageAreas[i].Clear();


  //
  // another ForEach does not compile, thus have to do the loop ourselves:
  //
  for (Int_t i=0;i<GetAverageSectors();i++)
    fAverageSectors[i].Clear();
  
  fTotalEntries = 0;
}

void MPedestalCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MPedestalPix *pix;
    while ((pix=(MPedestalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": ";
        *fLog << pix->GetPedestal() << " " << pix->GetPedestalRms() << endl;
    }
}

Float_t MPedestalCam::GetPedestalMin(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

Float_t MPedestalCam::GetPedestalMax(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

Bool_t MPedestalCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (GetSize() <= idx)
    return kFALSE;

  const Float_t ped      = (*this)[idx].GetPedestal();
  const Float_t rms      = (*this)[idx].GetPedestalRms();

  const Float_t pederr   = rms/TMath::Sqrt((Float_t)fTotalEntries);
  const Float_t rmserr   = rms/TMath::Sqrt((Float_t)fTotalEntries)/2.;

  switch (type)
    {
    case 0:
      val = ped;
      break;
    case 1:
      val = pederr;
      break;
    case 2:
      val = rms;
      break;
    case 3:
      val = rmserr;
      break;
    default:
      return kFALSE;
    }
  return kTRUE;
}

void MPedestalCam::DrawPixelContent(Int_t idx) const
{
  *fLog << warn << "MPedestalCam::DrawPixelContent - not available." << endl;
}
